import { GoogleGenAI, Type } from "@google/genai";
import { TicketType, AIAnalysisResult } from "../types";

// Initialize Gemini Client
const ai = new GoogleGenAI({ apiKey: process.env.API_KEY });

export const analyzeTicketMessage = async (rawMessage: string): Promise<AIAnalysisResult> => {
  try {
    const model = "gemini-2.5-flash";
    const prompt = `
      You are a helpful customer support assistant for a photography and digital services website named "ShayanPix".
      Analyze the following user message written in Persian.
      
      Tasks:
      1. Refine the text to be more polite, professional, and clear (keep it in Persian).
      2. Categorize the message into one of the following Ticket Types based on intent:
         - 'پشتیبانی فنی' (Technical Support)
         - 'مشاوره خرید' (Sales/Buying Advice)
         - 'امور مالی' (Billing/Finance)
         - 'سایر موارد' (Other)

      User Message: "${rawMessage}"
    `;

    const response = await ai.models.generateContent({
      model: model,
      contents: prompt,
      config: {
        responseMimeType: "application/json",
        responseSchema: {
          type: Type.OBJECT,
          properties: {
            refinedMessage: {
              type: Type.STRING,
              description: "The professional, corrected Persian version of the user's message."
            },
            suggestedType: {
              type: Type.STRING,
              enum: [
                TicketType.SUPPORT,
                TicketType.SALES,
                TicketType.BILLING,
                TicketType.OTHER
              ],
              description: "The most appropriate category for this ticket."
            }
          },
          required: ["refinedMessage", "suggestedType"]
        }
      }
    });

    const jsonText = response.text;
    if (!jsonText) {
      throw new Error("Empty response from AI");
    }

    const result = JSON.parse(jsonText) as AIAnalysisResult;
    return result;

  } catch (error) {
    console.error("Gemini Analysis Error:", error);
    // Fallback if AI fails: return original message and default type
    return {
      refinedMessage: rawMessage,
      suggestedType: TicketType.OTHER
    };
  }
};