import React, { useState } from 'react';
import { Send, Sparkles, User, Phone, Mail, FileText, CheckCircle2, Loader2, MessageSquare } from 'lucide-react';
import { TicketType, ContactMethod, TicketFormState } from '../types';
import { analyzeTicketMessage } from '../services/geminiService';
import { sendTicketToSupport } from '../services/emailService';
import { InputField } from './InputField';
import { SelectField } from './SelectField';

export const TicketForm: React.FC = () => {
  const [formData, setFormData] = useState<TicketFormState>({
    fullName: '',
    phoneNumber: '',
    email: '',
    ticketType: TicketType.SUPPORT,
    contactMethod: ContactMethod.SMS,
    message: ''
  });

  const [isAiLoading, setIsAiLoading] = useState(false);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitSuccess, setSubmitSuccess] = useState(false);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleAiEnhance = async () => {
    if (!formData.message || formData.message.length < 5) return;
    
    setIsAiLoading(true);
    try {
      const result = await analyzeTicketMessage(formData.message);
      setFormData(prev => ({
        ...prev,
        message: result.refinedMessage,
        ticketType: result.suggestedType
      }));
    } catch (error) {
      console.error("AI Enhance failed", error);
    } finally {
      setIsAiLoading(false);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);
    try {
      await sendTicketToSupport(formData);
      setSubmitSuccess(true);
    } catch (error) {
      alert("خطا در ارسال تیکت. لطفا مجددا تلاش کنید.");
    } finally {
      setIsSubmitting(false);
    }
  };

  if (submitSuccess) {
    return (
      <div className="bg-white rounded-2xl shadow-xl p-8 max-w-lg w-full text-center space-y-6 animate-fade-in">
        <div className="w-20 h-20 bg-green-100 text-green-600 rounded-full flex items-center justify-center mx-auto">
          <CheckCircle2 size={48} />
        </div>
        <div>
          <h2 className="text-2xl font-bold text-gray-800 mb-2">تیکت شما ثبت شد!</h2>
          <p className="text-gray-600">
            پیام شما با موفقیت برای تیم شایان پیکس ارسال شد.<br/>
            پاسخ به زودی به {formData.contactMethod} {formData.phoneNumber} ارسال خواهد شد.
          </p>
        </div>
        <button 
          onClick={() => {
            setSubmitSuccess(false);
            setFormData({ ...formData, message: '', ticketType: TicketType.SUPPORT });
          }}
          className="bg-indigo-600 text-white px-8 py-3 rounded-xl hover:bg-indigo-700 transition shadow-lg w-full"
        >
          ارسال تیکت جدید
        </button>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-2xl shadow-xl p-6 md:p-8 max-w-2xl w-full border border-gray-100">
      <div className="flex items-center gap-3 mb-8 border-b border-gray-100 pb-4">
        <div className="p-3 bg-indigo-50 text-indigo-600 rounded-xl">
          <MessageSquare size={28} />
        </div>
        <div>
          <h1 className="text-2xl font-bold text-gray-800">ارسال تیکت جدید</h1>
          <p className="text-sm text-gray-500">برای ارتباط با شایان پیکس فرم زیر را تکمیل کنید</p>
        </div>
      </div>

      <form onSubmit={handleSubmit} className="space-y-6">
        {/* Name and Phone */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <InputField
            label="نام و نام خانوادگی"
            name="fullName"
            placeholder="مثال: علی محمدی"
            value={formData.fullName}
            onChange={handleChange}
            required
            icon={<User size={18} />}
          />
          <InputField
            label="شماره موبایل"
            name="phoneNumber"
            type="tel"
            placeholder="0912..."
            value={formData.phoneNumber}
            onChange={handleChange}
            required
            dir="ltr"
            className="text-right"
            icon={<Phone size={18} />}
          />
        </div>

        {/* Email */}
        <InputField
            label="آدرس ایمیل (اختیاری)"
            name="email"
            type="email"
            placeholder="example@gmail.com"
            value={formData.email}
            onChange={handleChange}
            dir="ltr"
            className="text-right"
            icon={<Mail size={18} />}
          />

        {/* Type and Method */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <SelectField
            label="موضوع درخواست"
            name="ticketType"
            value={formData.ticketType}
            onChange={handleChange}
            options={Object.values(TicketType)}
          />

          <div className="flex flex-col gap-3">
            <label className="text-sm font-medium text-gray-700">نحوه اطلاع‌رسانی پاسخ</label>
            <div className="flex gap-4">
              {[ContactMethod.SMS, ContactMethod.PHONE].map((method) => (
                <label key={method} className={`
                  flex-1 flex items-center justify-center gap-2 p-3 rounded-lg border cursor-pointer transition-all
                  ${formData.contactMethod === method 
                    ? 'border-indigo-500 bg-indigo-50 text-indigo-700 ring-2 ring-indigo-200' 
                    : 'border-gray-200 hover:border-indigo-300 text-gray-600'}
                `}>
                  <input
                    type="radio"
                    name="contactMethod"
                    value={method}
                    checked={formData.contactMethod === method}
                    onChange={handleChange}
                    className="hidden"
                  />
                  <span className="text-sm font-medium">{method}</span>
                </label>
              ))}
            </div>
          </div>
        </div>

        {/* Message Area with AI */}
        <div className="space-y-2">
          <div className="flex justify-between items-center">
            <label className="text-sm font-medium text-gray-700 flex items-center gap-2">
              <FileText size={16} />
              متن درخواست
            </label>
            {formData.message.length > 5 && (
              <button
                type="button"
                onClick={handleAiEnhance}
                disabled={isAiLoading}
                className="text-xs flex items-center gap-1.5 text-indigo-600 bg-indigo-50 hover:bg-indigo-100 px-3 py-1.5 rounded-full transition-colors disabled:opacity-50"
              >
                {isAiLoading ? <Loader2 size={12} className="animate-spin"/> : <Sparkles size={12} />}
                {isAiLoading ? 'در حال بازنویسی...' : 'بازنویسی با هوش مصنوعی'}
              </button>
            )}
          </div>
          <textarea
            name="message"
            rows={5}
            className="w-full px-4 py-3 rounded-xl border border-gray-300 focus:ring-4 focus:ring-indigo-200 focus:border-indigo-500 focus:outline-none transition-all resize-none text-gray-700 leading-relaxed"
            placeholder="لطفا جزئیات درخواست خود را بنویسید..."
            value={formData.message}
            onChange={handleChange}
            required
          />
          <p className="text-xs text-gray-400 text-left">
            ارسال به: Shayanpix@gmail.com
          </p>
        </div>

        {/* Submit Button */}
        <button
          type="submit"
          disabled={isSubmitting}
          className="w-full bg-gradient-to-r from-indigo-600 to-purple-600 text-white font-bold py-3.5 rounded-xl hover:shadow-lg hover:from-indigo-700 hover:to-purple-700 transition-all transform hover:-translate-y-0.5 active:translate-y-0 disabled:opacity-70 disabled:cursor-not-allowed flex items-center justify-center gap-2"
        >
          {isSubmitting ? (
            <>
              <Loader2 className="animate-spin" size={20} />
              در حال ارسال...
            </>
          ) : (
            <>
              ارسال تیکت
              <Send size={20} className="rotate-180" />
            </>
          )}
        </button>
      </form>
    </div>
  );
};